/*
 *  linux/drivers/char/vga.c
 *
 *  Copyright (C) 1991, 1992  Linus Torvalds
 *			1995  Jay Estabrook
 */

/*
 *	vga.c
 *
 * This module exports the console low-level io support for VGA
 *
 * #ifndef CONFIG_MULTIMON
 *     'int con_get_font(char *data)'
 *     'int con_set_font(char *data, int ch512)'
 *     'int con_adjust_height(int fontheight)'
 *
 *     'int con_get_cmap(char *)'
 *     'int con_set_cmap(char *)'
 *
 *     'int reset_palette(int currcons)'
 *     'void set_palette(void)'
 * #else
 *     'int vga_con_get_font(char *data)'
 *     'int vga_con_set_font(char *data, int ch512)'
 *     'int vga_con_adjust_height(int mon, int fontheight)'
 *
 *     'int vga_con_get_cmap(char *)'
 *     'int vga_con_set_cmap(char *)'
 *
 *     'int vga_reset_palette(int currcons)'
 *     'void vga_set_palette(int currcons)'
 * #endif  CONFIG_MULTIMON
 *
 * User definable mapping table and font loading by Eugene G. Crosser,
 * <crosser@pccross.msk.su>
 *
 * Improved loadable font/UTF-8 support by H. Peter Anvin 
 * Feb-Sep 1995 <peter.anvin@linux.org>
 *
 * Colour palette handling, by Simon Tatham
 * 17-Jun-95 <sgt20@cam.ac.uk>
 *
 * if 512 char mode is already enabled don't re-enable it,
 * because it causes screen to flicker, by Mitja Horvat
 * 5-May-96 <mitja.horvat@guest.arnes.si>
 *
 */

#include "../../include/linux/autoconf.h"
#include <linux/sched.h>
#include <linux/timer.h>
#include <linux/interrupt.h>
#include <linux/tty.h>
#include <linux/tty_flip.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/kd.h>
#include <linux/malloc.h>
#include <linux/major.h>
#include <linux/mm.h>
#include <linux/ioport.h>

#include <asm/io.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <asm/bitops.h>

#include "kbd_kern.h"
#include "vt_kern.h"
#include "consolemap.h"
#include "selection.h"
#include "console_struct.h"
#ifdef CONFIG_MULTIMON
#include "multimon.h"
#endif /* CONFIG_MULTIMON */

#define CAN_LOAD_EGA_FONTS    /* undefine if the user must not do this */
#define CAN_LOAD_PALETTE      /* undefine if the user must not do this */

#define dac_reg (0x3c8)
#define dac_val (0x3c9)

#ifndef CONFIG_MULTIMON

#else
void vga_set_palette (int currcons);
void vga__set_origin (int currcons, unsigned short offset);
void vga_hide_cursor (int currcons);
void mono_hide_cursor (int currcons);
void vga_set_cursor (int currcons);
void mono_set_cursor (int currcons);
void vga_get_scrmem (int currcons);
void vga_set_scrmem (int currcons, long offset);
void hga_set_scrmem (int currcons, long offset);
int vga_set_get_font (int mon, char * arg, int, int);
int vga_set_get_cmap (int mon, unsigned char * arg, int);
int vga_con_adjust_height (int mon, unsigned long fontheight);
		       
#endif /* CONFIG_MULTIMON */
void
#ifndef CONFIG_MULTIMON
set_palette (void)
#else
vga_set_palette (int currcons)
#endif /* CONFIG_MULTIMON */
{
	int i, j ;

#ifndef CONFIG_MULTIMON
	if (video_type != VIDEO_TYPE_VGAC || console_blanked ||
	    vt_cons[fg_console]->vc_mode == KD_GRAPHICS)
#else
   /*      (video_type != VIDEO_TYPE_VGAC) */
	if (console_blanked || vt_cons[current_vc]->vc_mode == KD_GRAPHICS)
#endif /* CONFIG_MULTIMON */
		return ;

	for (i=j=0; i<16; i++) {
		outb_p (color_table[i], dac_reg) ;
		outb_p (vc_cons[fg_console].d->vc_palette[j++]>>2, dac_val) ;
		outb_p (vc_cons[fg_console].d->vc_palette[j++]>>2, dac_val) ;
		outb_p (vc_cons[fg_console].d->vc_palette[j++]>>2, dac_val) ;
	}
}

void
#ifndef CONFIG_MULTIMON
__set_origin(unsigned short offset)
#else
vga__set_origin(int currcons, unsigned short offset)
#endif /* CONFIG_MULTIMON */
{
	unsigned long flags;

	clear_selection();

	save_flags(flags); cli();
	__origin = offset;
	outb_p(12, video_port_reg);
	outb_p(offset >> 8, video_port_val);
	outb_p(13, video_port_reg);
	outb_p(offset, video_port_val);
	restore_flags(flags);
}

/*
 * Put the cursor just beyond the end of the display adaptor memory.
 */
void
#ifndef CONFIG_MULTIMON
hide_cursor(void)
#else
vga_hide_cursor(int currcons)
#endif /* CONFIG_MULTIMON */
{
  /* This is inefficient, we could just put the cursor at 0xffff,
     but perhaps the delays due to the inefficiency are useful for
     some hardware... */
	outb_p(14, video_port_reg);
	outb_p(0xff&((video_mem_term-video_mem_base)>>9), video_port_val);
	outb_p(15, video_port_reg);
	outb_p(0xff&((video_mem_term-video_mem_base)>>1), video_port_val);
}

void
#ifndef CONFIG_MULTIMON
set_cursor(int currcons)
#else
vga_set_cursor(int currcons)
#endif /* CONFIG_MULTIMON */
{
	unsigned long flags;

#ifndef CONFIG_MULTIMON
	if (currcons != fg_console || console_blanked || vcmode == KD_GRAPHICS)
#else
	if ((currcons != current_vc) || console_blanked || vcmode == KD_GRAPHICS)
#endif /* CONFIG_MULTIMON */
		return;
	if (__real_origin != __origin)
#ifndef CONFIG_MULTIMON
		__set_origin(__real_origin);
#else
		vga__set_origin(currcons, __real_origin);
#endif /* CONFIG_MULTIMON */
	save_flags(flags); cli();
	if (deccm) {
		outb_p(14, video_port_reg);
		outb_p(0xff&((pos-video_mem_base)>>9), video_port_val);
		outb_p(15, video_port_reg);
		outb_p(0xff&((pos-video_mem_base)>>1), video_port_val);
	} else
#ifndef CONFIG_MULTIMON
		hide_cursor();
#else
		vga_hide_cursor(currcons);
#endif /* CONFIG_MULTIMON */
	restore_flags(flags);
}

#ifdef CONFIG_MULTIMON
#ifdef MDA_CURSOR_BUG
  void
  mono_hide_cursor(int currcons)
  { /* Hide the cursor by setting the top and bottom scan */
     outb_p(0x0a, video_port_reg);
     outb_p(0x20, video_port_val);
     outb_p(0x0b, video_port_reg);
     outb_p(0x20, video_port_val);
  }
 
  void
  mono_set_cursor(int currcons)
  {
        unsigned long flags;
 
        if ((currcons != current_vc) || console_blanked || vcmode == KD_GRAPHICS)
                return;
        save_flags(flags); cli();
        if (deccm) {
           outb_p(14, video_port_reg);
           outb_p(0xff&((pos-video_mem_base)>>9), video_port_val);
           outb_p(15, video_port_reg);
           outb_p(0xff&((pos-video_mem_base)>>1), video_port_val);
           outb_p(0x0a, video_port_reg);
           outb_p(0x0b, video_port_val);
           outb_p(0x0b, video_port_reg);
           outb_p(0x0c, video_port_val);
        }
          else mono_hide_cursor(currcons);
        restore_flags(flags);
  }
#endif
          
#endif /* CONFIG_MULTIMON */
unsigned long
#ifndef CONFIG_MULTIMON
con_type_init(unsigned long kmem_start, const char **display_desc)
#else
vga_con_type_init(unsigned long kmem_start, struct monitor_data *m)
#endif /* CONFIG_MULTIMON */
{
#ifdef CONFIG_MULTIMON
    int i;

    m->m_video_num_lines = 25;
    m->m_video_num_columns = 80;
    m->set_org = vga__set_origin;
    m->hide_cur = vga_hide_cursor;
    m->set_cur = vga_set_cursor;
    m->set_scr = vga_set_scrmem;
    m->get_scr = vga_get_scrmem;
    if (m->m_video_type == 0) {
#endif /* CONFIG_MULTIMON */
	if (ORIG_VIDEO_MODE == 7)	/* Is this a monochrome display? */
	{
#ifndef CONFIG_MULTIMON
		video_mem_base = 0xb0000;
		video_port_reg = 0x3b4;
		video_port_val = 0x3b5;
#endif /* CONFIG_MULTIMON */
		if ((ORIG_VIDEO_EGA_BX & 0xff) != 0x10)
#ifndef CONFIG_MULTIMON
		{
			video_type = VIDEO_TYPE_EGAM;
			video_mem_term = 0xb8000;
			*display_desc = "EGA+";
			request_region(0x3b0,16,"ega");
		}
		else
		{
			video_type = VIDEO_TYPE_MDA;
			video_mem_term = 0xb2000;
			*display_desc = "*MDA";
			request_region(0x3b0,12,"mda");
			request_region(0x3bf, 1,"mda");
		}
#else
			m->m_video_type = VIDEO_TYPE_EGAM;
#ifdef CONFIG_HGA
			else m->m_video_type = VIDEO_TYPE_MDA;
#endif CONFIG_HGA
#endif /* CONFIG_MULTIMON */
	}
	else				/* If not, it is color. */
	{
#ifndef CONFIG_MULTIMON
		can_do_color = 1;
		video_mem_base = 0xb8000;
		video_port_reg	= 0x3d4;
		video_port_val	= 0x3d5;
#endif /* CONFIG_MULTIMON */
		if ((ORIG_VIDEO_EGA_BX & 0xff) != 0x10)
		{
#ifndef CONFIG_MULTIMON
			int i ;

			video_mem_term = 0xc0000;

#endif /* CONFIG_MULTIMON */
			if (!ORIG_VIDEO_ISVGA) {
#ifndef CONFIG_MULTIMON
				video_type = VIDEO_TYPE_EGAC;
				*display_desc = "EGA";
				request_region(0x3c0,32,"ega");
#else
				m->m_video_type = VIDEO_TYPE_EGAC;
#endif /* CONFIG_MULTIMON */
			} else {
#ifndef CONFIG_MULTIMON
				video_type = VIDEO_TYPE_VGAC;
				*display_desc = "VGA+";
				request_region(0x3c0,32,"vga+");

#ifdef VGA_CAN_DO_64KB
				/*
				 * get 64K rather than 32K of video RAM.
				 * This doesn't actually work on all "VGA"
				 * controllers (it seems like setting MM=01
				 * and COE=1 isn't necessarily a good idea)
				 */
				video_mem_base = 0xa0000 ;
				video_mem_term = 0xb0000 ;
				outb_p (6, 0x3ce) ;
				outb_p (6, 0x3cf) ;
#endif

				/*
				 * Normalise the palette registers, to point
				 * the 16 screen colours to the first 16
				 * DAC entries.
				 */

				for (i=0; i<16; i++) {
					inb_p (0x3da) ;
					outb_p (i, 0x3c0) ;
					outb_p (i, 0x3c0) ;
				}
				outb_p (0x20, 0x3c0) ;

				/* now set the DAC registers back to their
				 * default values */

				for (i=0; i<16; i++) {
					outb_p (color_table[i], 0x3c8) ;
					outb_p (default_red[i], 0x3c9) ;
					outb_p (default_grn[i], 0x3c9) ;
					outb_p (default_blu[i], 0x3c9) ;
				}
#else
				m->m_video_type = VIDEO_TYPE_VGAC;
#endif /* CONFIG_MULTIMON */
			}
		}
		else
		{
#ifndef CONFIG_MULTIMON
			video_type = VIDEO_TYPE_CGA;
			video_mem_term = 0xba000;
			*display_desc = "*CGA";
			request_region(0x3d4,2,"cga");
#else
			m->m_video_type = VIDEO_TYPE_CGA;
#endif /* CONFIG_MULTIMON */
		}
#ifdef CONFIG_MULTIMON
	    }
#endif /* CONFIG_MULTIMON */
	}
#ifndef CONFIG_MULTIMON
	return kmem_start;
#else
       switch (m->m_video_type)  {
           case VIDEO_TYPE_EGAM :
              if (m->m_video_mem_base == 0)
                m->m_video_mem_base = 0xb0000;
              if (m->m_video_port_base == 0)
                m->m_video_port_base = 0x3b0;
              m->m_video_mem_term = m->m_video_mem_base + 0x8000;
              m->m_video_port_reg = m->m_video_port_base + 0x04;
              m->m_video_port_val = m->m_video_port_base + 0x05;
              m->m_can_do_color = 0;
              strcpy(m->m_display_desc,"EGA+");
              request_region(m->m_video_port_base,16,"ega");
              break;

           case VIDEO_TYPE_MDA :
              if (m->m_video_mem_base == 0)
                m->m_video_mem_base = 0xb0000;
              if (m->m_video_port_base == 0)
                m->m_video_port_base = 0x3b0;
              m->m_video_mem_term = m->m_video_mem_base + 0x2000;
              m->m_video_port_reg = m->m_video_port_base + 0x04;
              m->m_video_port_val = m->m_video_port_base + 0x05;
              m->m_can_do_color = 0;
 #ifdef MDA_CURSOR_BUG
              m->set_cur = mono_set_cursor;
              m->hide_cur = mono_hide_cursor;
 #endif
              strcpy(m->m_display_desc,"*MDA");
              request_region(m->m_video_port_base,12,"mda");
              request_region(m->m_video_port_base + 0x0f,1,"mda");
              break;

           case VIDEO_TYPE_EGAC :
              if (m->m_video_mem_base == 0)
                m->m_video_mem_base = 0xb8000;
              if (m->m_video_port_base == 0)
                m->m_video_port_base = 0x3c0;
              m->m_video_mem_term = m->m_video_mem_base + 0x8000;
              m->m_video_port_reg = m->m_video_port_base + 0x14;
              m->m_video_port_val = m->m_video_port_base + 0x15;
              m->m_can_do_color = 1;
              strcpy(m->m_display_desc,"EGA");
              request_region(m->m_video_port_base,32,"ega");
                      break;

           case VIDEO_TYPE_VGAC :
              if (m->m_video_mem_base == 0)
                m->m_video_mem_base = 0xb8000;
              if (m->m_video_port_base == 0)
                m->m_video_port_base = 0x3c0;
              m->m_video_mem_term = m->m_video_mem_base + 0x8000;
              m->m_video_port_reg = m->m_video_port_base + 0x14;
              m->m_video_port_val = m->m_video_port_base + 0x15;
              m->m_can_do_color = 1;
              m->set_pal = vga_set_palette;
              m->sg_font = vga_set_get_font;
              m->sg_cmap = vga_set_get_cmap;
              m->con_adjust = vga_con_adjust_height;
              strcpy(m->m_display_desc,"VGA+");
#ifdef VGA_CAN_DO_64KB
                /*
                 * get 64K rather than 32K of video RAM.
                 * This doesn't actually work on all "VGA"
                 * controllers (it seems like setting MM=01
                 * and COE=1 isn't necessarily a good idea)
                 */
               m->m_video_mem_base = 0xa0000 ;
               m->m_video_mem_term = 0xb0000 ;
               outb_p (6, m->m_video_port_base + 0xe) ;
               outb_p (6, m->m_video_port_base + 0xf) ;
#endif

                 /* normalise the palette registers, to point the
                  * 16 screen colours to the first 16 DAC entries */
               for (i=0; i<16; i++) {
                        inb_p (m->m_video_port_base + 0x1a) ;
                        outb_p (i, m->m_video_port_base) ;
                        outb_p (i, m->m_video_port_base) ;
               }
               outb_p (0x20, m->m_video_port_base) ;

                 /* now set the DAC registers back to their default
                * values */
               for (i=0; i<16; i++) {
                        outb_p (color_table[i], m->m_video_port_base + 8) ;
                        outb_p (default_red[i], m->m_video_port_base + 9) ;
                        outb_p (default_grn[i], m->m_video_port_base + 9) ;
                        outb_p (default_blu[i], m->m_video_port_base + 9) ;
               }
               request_region(m->m_video_port_base,32,"vga");
               break;
 
            case VIDEO_TYPE_CGA :
               if (m->m_video_mem_base == 0)
                 m->m_video_mem_base = 0xb8000;
               if (m->m_video_port_base == 0)
                 m->m_video_port_base = 0x3d4;
               m->m_video_mem_term = m->m_video_mem_base + 0x2000;
               m->m_video_port_reg = m->m_video_port_base;
               m->m_video_port_val = m->m_video_port_base + 0x01;
               m->m_can_do_color = 1;
               strcpy(m->m_display_desc,"*CGA");
               request_region(m->m_video_port_base,2,"cga");
        }
        m->m_video_size_row = m->m_video_num_columns * 2;
          m->m_video_screen_size = m->m_video_size_row * m->m_video_num_lines;
    return kmem_start;
#endif /* CONFIG_MULTIMON */
}

void
#ifndef CONFIG_MULTIMON
get_scrmem(int currcons)
#else
vga_get_scrmem(int currcons)
#endif /* CONFIG_MULTIMON */
{
	memcpyw((unsigned short *)vc_scrbuf[currcons],
		(unsigned short *)origin, video_screen_size);
	origin = video_mem_start = (unsigned long)vc_scrbuf[currcons];
	scr_end = video_mem_end = video_mem_start + video_screen_size;
	pos = origin + y*video_size_row + (x<<1);
}

#ifndef CONFIG_MULTIMON
void set_scrmem(int currcons, long offset)
{
#else
void vga_set_scrmem(int currcons, long offset);
void hga_set_scrmem (int currcons, long offset)
{
#endif /* CONFIG_MULTIMON */
#ifdef CONFIG_HGA
  /* This works with XFree86 1.2, 1.3 and 2.0
     This code could be extended and made more generally useful if we could
     determine the actual video mode. It appears that this should be
     possible on a genuine Hercules card, but I (WM) haven't been able to
     read from any of the required registers on my clone card.
     */
	/* This code should work with Hercules and MDA cards. */
#ifndef CONFIG_MULTIMON
	if (video_type == VIDEO_TYPE_MDA)
	  {
	    if (vcmode == KD_TEXT)
	      {
#else
	if (vcmode == KD_TEXT)
	    {
#endif /* CONFIG_MULTIMON */
		/* Ensure that the card is in text mode. */
		int	i;
		static char herc_txt_tbl[12] = {
		  0x61,0x50,0x52,0x0f,0x19,6,0x19,0x19,2,0x0d,0x0b,0x0c };
		outb_p(0, 0x3bf);  /* Back to power-on defaults */
		outb_p(0, 0x3b8);  /* Blank the screen, select page 0, etc */
		for ( i = 0 ; i < 12 ; i++ )
		  {
		    outb_p(i, 0x3b4);
		    outb_p(herc_txt_tbl[i], 0x3b5);
		  }
	      }
#define HGA_BLINKER_ON 0x20
#define HGA_SCREEN_ON  8
	    /* Make sure that the hardware is not blanked */
	    outb_p(HGA_BLINKER_ON | HGA_SCREEN_ON, 0x3b8);
#ifdef CONFIG_MULTIMON
	    vga_setscrmem(currcons, offset);
#endif /* CONFIG_MULTIMON */
	  }
#endif CONFIG_HGA

#ifdef CONFIG_MULTIMON
}

void vga_set_scrmem(int currcons, long offset)
{
#endif /* CONFIG_MULTIMON */
	if (video_mem_term - video_mem_base < offset + video_screen_size)
	  offset = 0;	/* strange ... */
	memcpyw((unsigned short *)(video_mem_base + offset),
		(unsigned short *) origin, video_screen_size);
	video_mem_start = video_mem_base;
	video_mem_end = video_mem_term;
	origin = video_mem_base + offset;
	scr_end = origin + video_screen_size;
	pos = origin + y*video_size_row + (x<<1);
	has_wrapped = 0;
}

/*
 * PIO_FONT support.
 *
 * The font loading code goes back to the codepage package by
 * Joel Hoffman (joel@wam.umd.edu). (He reports that the original
 * reference is: "From: p. 307 of _Programmer's Guide to PC & PS/2
 * Video Systems_ by Richard Wilton. 1987.  Microsoft Press".)
 *
 * Change for certain monochrome monitors by Yury Shevchuck
 * (sizif@botik.yaroslavl.su).
 */

#define colourmap ((char *)0xa0000)
/* Pauline Middelink <middelin@polyware.iaf.nl> reports that we
   should use 0xA0000 for the bwmap as well.. */
#define blackwmap ((char *)0xa0000)
#define cmapsz 8192
#define attrib_port (0x3c0)
#define seq_port_reg (0x3c4)
#define seq_port_val (0x3c5)
#define gr_port_reg (0x3ce)
#define gr_port_val (0x3cf)

int
#ifndef CONFIG_MULTIMON
set_get_font(char * arg, int set, int ch512)
#else
vga_set_get_font(int mon, char * arg, int set, int ch512)
#endif /* CONFIG_MULTIMON */
{
#ifdef CAN_LOAD_EGA_FONTS
	static int ch512enabled = 0;
	int i;
	char *charmap;
	int beg;
#ifdef CONFIG_MULTIMON
	int currcons = monitor[mon].m_current_vc;
#endif /* CONFIG_MULTIMON */
	unsigned short video_port_status = video_port_reg + 6;
	int font_select = 0x00;

	/* no use to "load" CGA... */

	if (video_type == VIDEO_TYPE_EGAC || video_type == VIDEO_TYPE_VGAC) {
		charmap = colourmap;
		beg = 0x0e;
#ifdef VGA_CAN_DO_64KB
		if (video_type == VIDEO_TYPE_VGAC)
			beg = 0x06;
#endif VGA_CAN_DO_64KB
	} else if (video_type == VIDEO_TYPE_EGAM) {
		charmap = blackwmap;
		beg = 0x0a;
	} else
		return -EINVAL;
	
	if (arg)
	  {
	    i = verify_area(set ? VERIFY_READ : VERIFY_WRITE, (void *)arg,
			    ch512 ? 2*cmapsz : cmapsz);
	    if (i)
	      return i;
	  }
	else
	  ch512 = 0;		/* Default font is always 256 */

#ifdef BROKEN_GRAPHICS_PROGRAMS
	/*
	 * All fonts are loaded in slot 0 (0:1 for 512 ch)
	 */

	if (!arg)
	  return -EINVAL;	/* Return to default font not supported */

	video_font_is_default = 0;
	font_select = ch512 ? 0x04 : 0x00;
#else	
	/*
	 * The default font is kept in slot 0 and is never touched.
	 * A custom font is loaded in slot 2 (256 ch) or 2:3 (512 ch)
	 */

	if (set)
	  {
	    video_font_is_default = !arg;
	    font_select = arg ? (ch512 ? 0x0e : 0x0a) : 0x00;
	  }

	if ( !video_font_is_default )
	  charmap += 4*cmapsz;
#endif

	cli();
	outb_p( 0x00, seq_port_reg );   /* First, the sequencer */
	outb_p( 0x01, seq_port_val );   /* Synchronous reset */
	outb_p( 0x02, seq_port_reg );
	outb_p( 0x04, seq_port_val );   /* CPU writes only to map 2 */
	outb_p( 0x04, seq_port_reg );
	outb_p( 0x07, seq_port_val );   /* Sequential addressing */
	outb_p( 0x00, seq_port_reg );
	outb_p( 0x03, seq_port_val );   /* Clear synchronous reset */

	outb_p( 0x04, gr_port_reg );    /* Now, the graphics controller */
	outb_p( 0x02, gr_port_val );    /* select map 2 */
	outb_p( 0x05, gr_port_reg );
	outb_p( 0x00, gr_port_val );    /* disable odd-even addressing */
	outb_p( 0x06, gr_port_reg );
	outb_p( 0x00, gr_port_val );    /* map start at A000:0000 */
	sti();
	
	if (arg)
	  {
	    if (set)
	      for (i=0; i<cmapsz ; i++)
		scr_writeb(get_user(arg + i), charmap + i);
	    else
	      for (i=0; i<cmapsz ; i++)
		put_user(scr_readb(charmap + i), arg + i);
	    
	    
	/*
	 * In 512-character mode, the character map is not contiguous if
	 * we want to remain EGA compatible -- which we do
	 */

	    if (ch512)
	      {
		charmap += 2*cmapsz;
		arg += cmapsz;
		if (set)
		  for (i=0; i<cmapsz ; i++)
		    *(charmap+i) = get_user(arg+i);
		else
		  for (i=0; i<cmapsz ; i++)
		    put_user(*(charmap+i), arg+i);
	      }
	  }
	
	cli();
	outb_p( 0x00, seq_port_reg );   /* First, the sequencer */
	outb_p( 0x01, seq_port_val );   /* Synchronous reset */
	outb_p( 0x02, seq_port_reg );
	outb_p( 0x03, seq_port_val );   /* CPU writes to maps 0 and 1 */
	outb_p( 0x04, seq_port_reg );
	outb_p( 0x03, seq_port_val );   /* odd-even addressing */
	if (set)
	  {
	    outb_p( 0x03, seq_port_reg ); /* Character Map Select */
	    outb_p( font_select, seq_port_val );
	  }
	outb_p( 0x00, seq_port_reg );
	outb_p( 0x03, seq_port_val );   /* clear synchronous reset */

	outb_p( 0x04, gr_port_reg );    /* Now, the graphics controller */
	outb_p( 0x00, gr_port_val );    /* select map 0 for CPU */
	outb_p( 0x05, gr_port_reg );
	outb_p( 0x10, gr_port_val );    /* enable even-odd addressing */
	outb_p( 0x06, gr_port_reg );
	outb_p( beg, gr_port_val );     /* map starts at b800:0 or b000:0 */

	/* if 512 char mode is already enabled don't re-enable it. */
	if ((set)&&(ch512!=ch512enabled))	/* attribute controller */
	  {
	    ch512enabled=ch512;
	    /* 256-char: enable intensity bit
	       512-char: disable intensity bit */
	    inb_p( video_port_status );	/* clear address flip-flop */
	    outb_p ( 0x12, attrib_port ); /* color plane enable register */
	    outb_p ( ch512 ? 0x07 : 0x0f, attrib_port );
	    /* Wilton (1987) mentions the following; I don't know what
	       it means, but it works, and it appears necessary */
	    inb_p( video_port_status );
	    outb_p ( 0x20, attrib_port );
	  }
	sti();

	return 0;
#else
	return -EINVAL;
#endif
}

/*
 * Adjust the screen to fit a font of a certain height
 *
 * Returns < 0 for error, 0 if nothing changed, and the number
 * of lines on the adjusted console if changed.
 */
int
#ifndef CONFIG_MULTIMON
con_adjust_height(unsigned long fontheight)
#else
vga_con_adjust_height(int mon, unsigned long fontheight)
#endif /* CONFIG_MULTIMON */
{
	int rows, maxscan;
	unsigned char ovr, vde, fsr, curs, cure;
#ifdef CONFIG_MULTIMON
   	int currcons = monitor[mon].m_current_vc;
#endif /* CONFIG_MULTIMON */

	if (fontheight > 32 || (video_type != VIDEO_TYPE_VGAC &&
	    video_type != VIDEO_TYPE_EGAC && video_type != VIDEO_TYPE_EGAM))
		return -EINVAL;

	if ( fontheight == video_font_height || fontheight == 0 )
		return 0;

	video_font_height = fontheight;

	rows = video_scan_lines/fontheight;	/* Number of video rows we end up with */
	maxscan = rows*fontheight - 1;		/* Scan lines to actually display-1 */

	/* Reprogram the CRTC for the new font size
	   Note: the attempt to read the overflow register will fail
	   on an EGA, but using 0xff for the previous value appears to
	   be OK for EGA text modes in the range 257-512 scan lines, so I
	   guess we don't need to worry about it.

	   The same applies for the spill bits in the font size and cursor
	   registers; they are write-only on EGA, but it appears that they
	   are all don't care bits on EGA, so I guess it doesn't matter. */

	cli();
	outb_p( 0x07, video_port_reg );		/* CRTC overflow register */
	ovr = inb_p(video_port_val);
	outb_p( 0x09, video_port_reg );		/* Font size register */
	fsr = inb_p(video_port_val);
	outb_p( 0x0a, video_port_reg );		/* Cursor start */
	curs = inb_p(video_port_val);
	outb_p( 0x0b, video_port_reg );		/* Cursor end */
	cure = inb_p(video_port_val);
	sti();

	vde = maxscan & 0xff;			/* Vertical display end reg */
	ovr = (ovr & 0xbd) +			/* Overflow register */
	      ((maxscan & 0x100) >> 7) +
	      ((maxscan & 0x200) >> 3);
	fsr = (fsr & 0xe0) + (fontheight-1);    /*  Font size register */
	curs = (curs & 0xc0) + fontheight - (fontheight < 10 ? 2 : 3);
	cure = (cure & 0xe0) + fontheight - (fontheight < 10 ? 1 : 2);

	cli();
	outb_p( 0x07, video_port_reg );		/* CRTC overflow register */
	outb_p( ovr, video_port_val );
	outb_p( 0x09, video_port_reg );		/* Font size */
	outb_p( fsr, video_port_val );
	outb_p( 0x0a, video_port_reg );		/* Cursor start */
	outb_p( curs, video_port_val );
	outb_p( 0x0b, video_port_reg );		/* Cursor end */
	outb_p( cure, video_port_val );
	outb_p( 0x12, video_port_reg );		/* Vertical display limit */
	outb_p( vde, video_port_val );
	sti();

	if ( rows == video_num_lines ) {
	  /* Change didn't affect number of lines -- no need to scare
	     the rest of the world */
	  return 0;
	}

#ifndef CONFIG_MULTIMON
	vc_resize(rows, 0);			/* Adjust console size */
#else
	vc_resize(mon, rows, 0);			/* Adjust console size */
#endif /* CONFIG_MULTIMON */

	return rows;
}

int
#ifndef CONFIG_MULTIMON
set_get_cmap(unsigned char * arg, int set) {
#else
vga_set_get_cmap(int mon, unsigned char * arg, int set) {
#endif /* CONFIG_MULTIMON */
#ifdef CAN_LOAD_PALETTE
	int i;
#ifdef CONFIG_MULTIMON
	int currcons = monitor[mon].m_current_vc;
#endif /* CONFIG_MULTIMON */
   
	/* no use to set colourmaps in less than colour VGA */

	if (video_type != VIDEO_TYPE_VGAC)
		return -EINVAL;

	i = verify_area(set ? VERIFY_READ : VERIFY_WRITE, (void *)arg, 16*3);
	if (i)
		return i;

	for (i=0; i<16; i++) {
		if (set) {
			default_red[i] = get_user(arg++) ;
			default_grn[i] = get_user(arg++) ;
			default_blu[i] = get_user(arg++) ;
		} else {
			put_user (default_red[i], arg++) ;
			put_user (default_grn[i], arg++) ;
			put_user (default_blu[i], arg++) ;
		}
	}
	if (set) {
		for (i=0; i<MAX_NR_CONSOLES; i++)
			if (vc_cons_allocated(i)) {
				int j, k ;
				for (j=k=0; j<16; j++) {
					vc_cons[i].d->vc_palette[k++] = default_red[j];
					vc_cons[i].d->vc_palette[k++] = default_grn[j];
					vc_cons[i].d->vc_palette[k++] = default_blu[j];
				}
			}
#ifndef CONFIG_MULTIMON
		set_palette() ;
#else
		set_palette(currcons);
#endif /* CONFIG_MULTIMON */
	}

	return 0;
#else
	return -EINVAL;
#endif
}
