/*****************************************************************************/
/* mod_auth_radius.c                                                         */
/*                                                                           */
/* Directives:                                                               */ 
/*                                                                           */
/*   AuthRadConfig keyword filename                                          */
/*     AuthRadConfig takes a keyword and a radius clients file, this clients */
/*         file is read by the user who runs apache (not the user apache     */
/*         runs as), typically root, for security reasons.                   */
/*         AuthRadConfig is only valid in one of the server .conf files.     */
/*                                                                           */
/*   AuthRadCache keyword size expiry                                        */
/*     AuthRadCache takes a keyword and a some caching parameters.  Size is  */
/*         the size of the cache in entries, and expiry is the number of     */
/*         seconds a cache entry is valid for.                               */
/*         AuthRadCache is only valid in one of the server .conf files.      */
/*                                                                           */
/*   AuthRadius keyword                                                      */
/*     AuthRadius is an .htaccess directive to enable radius authentication  */
/*         for the directory using the clients file specified with           */
/*         AuthRadConfig.                                                    */
/*                                                                           */
/* This code was gleaned from many hours of staring at Apache and RADIUS     */
/* source code and should follow their licensing schemes.  The Apache HTTP   */
/* server project (http://www.apache.org/) and Livingston RADIUS             */
/* (http://www.livingston.org/) both grant permission to use, copy, modify   */
/* and distribute source and binaries for any purpose and without fee.       */
/*                                                                           */
/* If you have any problems/comments email noc@trap.mtview.ca.us             */
/*                                                                           */
/*****************************************************************************/

#ifdef OSF1
#include <compat.h>
#endif
#include <apache-1.3/httpd.h>
#include <apache-1.3/http_config.h>
#include <apache-1.3/http_core.h>
#include <apache-1.3/http_main.h>
#include <apache-1.3/http_log.h>
#include <apache-1.3/http_protocol.h>
#include <signal.h>
#include <setjmp.h>

#define MAX_SERVER_LEN	128
#define MAX_SECRET_LEN	16
#define MAX_KEY_LEN	16
#define RECVTIMEOUT	6	/* in seconds */
#define RAD_RETRY	4

module radius_auth_module;

static const char *handle_AuthRadConfig(cmd_parms *cmd, void *mconfig, char *key,char *file);
static const char *handle_AuthRadCache(cmd_parms *cmd, void *mconfig, char *key,char *size, char *expiry);
static const char *handle_AuthRadius(cmd_parms *cmd, void *mconfig, char *key);
int rad_authenticate(char* username, char* password, char* hostname, char* secret, char seq);

command_rec radius_auth_cmds[] ={
  {"AuthRadConfig",handle_AuthRadConfig, NULL, RSRC_CONF, TAKE2,
    "AuthRadConfig <key> <filename> specify RADIUS clients file"},
  {"AuthRadCache",handle_AuthRadCache, NULL, RSRC_CONF, TAKE23,
    "AuthRadCache <key> <size> <expiry> set RADIUS authentication cache options"},
  {"AuthRadius",handle_AuthRadius, NULL, OR_AUTHCFG, TAKE1,
    "AuthRadius <key> require RADIUS authentication"},
  {NULL}
};

typedef struct rad_server {	/* A server and her secret... */
  char server[MAX_SERVER_LEN];
  char secret[MAX_SECRET_LEN];
  char seq;
  struct rad_server *next;
} rad_server;

typedef struct cache_entry {	/* A single cache entry */
  char name[MAX_SECRET_LEN];
  char pass[MAX_SECRET_LEN];
  int time;
} cache_entry;

typedef struct rad_method {	/* method - one set of servers, one cache */
  char key[MAX_KEY_LEN];
  rad_server *server;
  int size;
  int expiry;
  cache_entry *cache;
  struct rad_method *next;
} rad_method;

typedef struct {		/* dir-level configuration */
  int method;
} rad_config;

static pool *radius_pool=NULL;
static rad_method *radius_methods=NULL;


static void *create_radius_auth_config(pool *p, char *t) {
	return ap_pcalloc(p, sizeof(rad_config));
}

static void setup_pool() {
  if(radius_pool==NULL) {
    radius_pool=ap_make_sub_pool(NULL);
  }
}

static const char *handle_AuthRadConfig(cmd_parms *cmd, void *mconfig, char *key,char *file) {
  rad_method **method;
  rad_server **server;
  FILE *clientFile;
  char buffer[128];
  char entry[1280];
  char *q=entry;
  char *p,*r;

  setup_pool();
  if (strlen(key)>MAX_KEY_LEN) return "AuthRadConfig: key too long.\n";
  method=&radius_methods;
  while(*method) {
    if(strcmp(key,(*method)->key)==0) return "AuthRadConfig: key already defined.\n";
    method=&((*method)->next);
  }
  (*method)=ap_pcalloc(radius_pool,sizeof(rad_method));
  strcpy((*method)->key,key);
  server=&((*method)->server);
  clientFile=ap_pfopen(radius_pool, file, "r");
  if (clientFile) {
    while(fgets(buffer,sizeof(buffer),clientFile)) {
      if((p=strchr(buffer,'#'))!=NULL) *p='\0';
        if((r=strtok(buffer," \t"))!=NULL) {
    	p=strtok(NULL," \t\n");
    	(*server)=ap_pcalloc(radius_pool,sizeof(rad_server));
    	strncpy((*server)->server,r,MAX_SERVER_LEN);
    	strncpy((*server)->secret,p,MAX_SECRET_LEN);
    	server=&((*server)->next);
      }
    }
    *q=0;
    ap_pfclose(radius_pool,clientFile);
    return NULL;
  } else {
    return "AuthRadConfig: couldn\'t open file.\n";
  }
}

static const char *handle_AuthRadCache(cmd_parms *cmd, void *mconfig, char *key, char *size, char *expiry) {
  rad_method **method=&radius_methods;

  while(*method) {
    if(strcmp(key,(*method)->key)==0) {
      sscanf(size,"%d",&((*method)->size));
      sscanf(expiry,"%d",&((*method)->expiry));
      (*method)->cache=(cache_entry*)ap_pcalloc(radius_pool,sizeof(cache_entry)*(*method)->size);
      return NULL;
    }
    method=&((*method)->next);
  }
  return "AuthRadCache: key isn't defined.\n";
}

static const char *handle_AuthRadius(cmd_parms *cmd, void *mconfig, char *key) {
  rad_config *cfg=(rad_config*)mconfig;
  rad_method **method;
  int method_index=0;
  method=&radius_methods;
  while(*method) {
    method_index++;
    if(strcmp((*method)->key,key)==0) {
      cfg->method=method_index;
      return NULL;
    }
    method=&((*method)->next);
  }
  return "AuthRadius: key isn't defined.\n";
}

/*
 * radius_authenticate_basic_user is supposed to go make the real radius query.
 * on auth success it should return zero and set *errstr to null.
 * on error it should return non-zero and set errstr to a meaningful
 * error string.
 */
static int radius_authenticate_basic_user(request_rec *r) {
	rad_config *dcfg;
	conn_rec *c = r->connection;
	char *sent_pw;
	rad_method **method;
	rad_server **server;
	char errstr[MAX_STRING_LEN];
	int res,method_index,try=RAD_RETRY;
	cache_entry *cachespot=NULL,*oldest=NULL;
	int i=0,age=0x7FFFFFFF;;

	if (r->per_dir_config == NULL) return DECLINED;
	dcfg = (rad_config *)
		ap_get_module_config(r->per_dir_config, &radius_auth_module);
	if (dcfg->method == 0) return DECLINED;
	if((res = ap_get_basic_auth_pw (r, (const char**)&sent_pw))) return res;

	res=-2;
	method=&radius_methods;
	method_index=dcfg->method;
	while(*method && --method_index) method=&((*method)->next);
	while(i<(*method)->size) {
	  if((*method)->cache[i].time+(*method)->expiry < (int)time(NULL))
	    memset((*method)->cache+i,0,sizeof(cache_entry));
	  if(strcmp((*method)->cache[i].name,c->user)==0) {
	    cachespot=(*method)->cache+i;
	    if(strcmp((*method)->cache[i].pass,sent_pw)==0) {
              time((time_t*)&((*method)->cache[i].time));
              return OK;
	    }
	  } else {
	    if(age > (*method)->cache[i].time) {
	      age=(*method)->cache[i].time;
	      oldest=(*method)->cache+i;
	    }
	  }
	  i++;
	}
	do{
	  server=&((*method)->server);
	  while(*server &&
	        ((res = rad_authenticate(c->user,sent_pw,
	        (*server)->server, (*server)->secret,
	        ++((*server)->seq))) >= 0)) {
	    server=&((*server)->next);
	  }	// hit each server
	}while(--try && res >=0);
	if(res==-1) {
	  if(cachespot==NULL) cachespot=oldest;
	  if(oldest!=NULL) {
	    strncpy(cachespot->name,c->user,MAX_SECRET_LEN);
	    strncpy(cachespot->pass,sent_pw,MAX_SECRET_LEN);
	    time((time_t*)&(cachespot->time));
	  }
	} else {
	  if(cachespot) memset(cachespot,0,sizeof(cache_entry));
	}
	if (res==-1) return OK;
	strcpy(errstr,strerror(res));
	ap_log_reason (errstr, r->uri, r);
	ap_note_basic_auth_failure (r);
	return AUTH_REQUIRED;
}

// ###########################################
module radius_auth_module = {
  STANDARD_MODULE_STUFF,
  NULL,					/* module initializer */
  create_radius_auth_config,		/* per-directory config creator */
  NULL,					/* dir config merger */
  NULL,					/* server config creator */
  NULL,					/* server config merger */
  radius_auth_cmds,			/* command table */
  NULL,					/* [7] list of handlers */
  NULL,					/* [2] filename-to-URI translation */
  radius_authenticate_basic_user,	/* [5] check/validate user_id */
  NULL,					/* [6] check user_id is valid *here* */
  NULL,					/* [4] check access by host address */
  NULL,					/* [7] MIME type checker/setter */
  NULL,					/* [8] fixups */
  NULL,					/* [10] logger */
  NULL,					/* [3] header parser */
  NULL,					/* process initializer */
  NULL,					/* process exit/cleanup */
  NULL					/* [1] post read_request handling */
};

/*===================== Actual RADUIS related code =====================*/
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <stdio.h>
#include <netdb.h>

#include <apache-1.3/ap_md5.h>
void md5_calc(unsigned char *output, unsigned char *input, unsigned int inlen) {
  AP_MD5_CTX context;
  ap_MD5Init(&context);
  ap_MD5Update(&context, input, inlen);
  ap_MD5Final(output, &context);
}

#include "radius.h"

void random_vector(u_char *vector);
int result_recv(AUTH_HDR *auth, char *secret, unsigned char *vector,
		unsigned char seq_nbr);

#define MAXPWNAM 20
#define MAXPASS 16

u_char recv_buffer[4096];
u_char send_buffer[4096];
int sockfd;
u_char vector[AUTH_VECTOR_LEN];
u_char secret_key[128];

    
/********************************************************************
 * rad_authenticate
 * Authenticate user on radius server
 * Return : -1 - User authenticated
 *          -2 - Access Denied
 *          errno - Error Occured
 *******************************************************************/
int rad_authenticate(char* username, char* password,
		     char* hostname, char* secret, char seq) {
    int salen;
    int result;
    struct sockaddr salocal;
    struct sockaddr saremote;
    struct sockaddr_in *sin;
    struct hostent *hp;
    struct servent *svp;
    u_short svc_port;
    AUTH_HDR *auth;
    u_char passbuf[AUTH_PASS_LEN];
    u_char md5buf[256];
    u_char *oldvector;
    UINT4 auth_ipaddr;
    u_short local_port;
    int total_length;
    u_char *ptr;
    int length;
    int secretlen;
    int i;

    strcpy(secret_key,secret);
    svp = getservbyname("radius", "udp");
    if (svp == (struct servent *)0) {
        return ENOPROTOOPT;
    }
  
    svc_port = ntohs((u_short)svp->s_port);

    /* Get the IP address of the authentication server */
    if((hp = gethostbyname(hostname)) == (struct hostent *)NULL) {
	return EDESTADDRREQ;
    }
    auth_ipaddr = (ntohl(*(UINT4 *)hp->h_addr));

    sockfd = socket(AF_INET, SOCK_DGRAM, 0);
    if (sockfd < 0) {
    	return errno;
    }

    sin = (struct sockaddr_in *)&salocal;
    memset ((char *)sin, '\0', sizeof(salocal));
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = INADDR_ANY;

    local_port = 1025;
    do {
	local_port++;
	sin->sin_port = htons((u_short)local_port);
    } while ((bind(sockfd, &salocal, sizeof(struct sockaddr_in)) < 0) && 
	     local_port < 64000);

    if (local_port >= 64000) {
	close(sockfd);
    	return ECONNREFUSED;
    }

    /* Build authenticaton request */
    auth = (AUTH_HDR *)send_buffer;
    auth->code = PW_AUTHENTICATION_REQUEST;
    auth->id = seq;
    random_vector(vector);
    memcpy(auth->vector, vector, AUTH_VECTOR_LEN);
    total_length = AUTH_HDR_LEN;
    ptr = auth->data;

    /* User Name */
    *ptr++ = PW_USER_NAME;
    length = strlen(username);
    if (length > MAXPWNAM)
	length = MAXPWNAM;
    *ptr++ = length + 2;
    memcpy(ptr, username, length);
    ptr += length;
    total_length += length + 2;

    /* Password */
    *ptr++ = PW_PASSWORD;
    *ptr++ = AUTH_PASS_LEN + 2;

    /* Encrypt the Password */
    length = strlen(password);
    if (length > MAXPASS)
	length = MAXPASS;
    memset(passbuf, 0, AUTH_PASS_LEN);
    memcpy(passbuf, password, length);
    /* Calculate the MD5 Digest */
    secretlen = strlen(secret_key);
    strcpy(md5buf, secret_key);
    memcpy(md5buf + secretlen, auth->vector, AUTH_VECTOR_LEN);
    md5_calc(ptr, md5buf, secretlen + AUTH_VECTOR_LEN);
    oldvector = ptr;

    /* Xor the password into the MD5 digest */
    for (i=0; i < AUTH_PASS_LEN; i++)
	*ptr++ ^= passbuf[i];
    total_length += AUTH_PASS_LEN + 2;


    *ptr++ = PW_CLIENT_ID;
    *ptr++ = 6; /* Length */
    *ptr++ = 196;
    *ptr++ = 14;
    *ptr++ = 80;
    *ptr++ = 129;
    total_length += 6;

    *ptr++ = PW_CLIENT_PORT_ID;

    *ptr++ = 6; /* Length */
    *ptr++ = 1;
    *ptr++ = 2;
    *ptr++ = 3;
    *ptr++ = 4;
    total_length += 6;    


    auth->length = htonl(total_length);

    sin = (struct sockaddr_in *)&saremote;
    memset((char *)sin, '\0', sizeof(saremote));
    sin->sin_family = AF_INET;
    sin->sin_addr.s_addr = htonl(auth_ipaddr);
    sin->sin_port = htons(svc_port);
    
    sendto(sockfd,(char *)auth,(int)total_length,(int)0,
	    &saremote, sizeof(struct sockaddr_in));

    salen = sizeof(saremote);
    {
	struct  sigaction sa,osa;
	sigjmp_buf jmpbuf;
	void handler(int handler_sig) { siglongjmp(jmpbuf,1); };

	sa.sa_handler=&handler;
	sigemptyset(&sa.sa_mask);
	sa.sa_flags=0;
	sigaction(SIGALRM,&sa,&osa);
	if(sigsetjmp(jmpbuf,0)) {
		sigaction(SIGALRM,&osa,NULL);
		return ETIMEDOUT;
	}
	alarm(RECVTIMEOUT);
	result = recvfrom(sockfd,(char *)recv_buffer, 
		      (int)sizeof(recv_buffer),
		      (int)0, & saremote, & salen);
	alarm(0);
	sigaction(SIGALRM,&osa,NULL);
    }
    close(sockfd);
    if (result > 0) {
	return result_recv((AUTH_HDR*)recv_buffer, secret, auth->vector,
			   auth->id);
    }
    return errno;
}

/*****************************************************************
 * result_recv
 * Return : -1 - User authenticated
 *          -2 - Access Denied
 *          errno - Error Occured
 */
int result_recv(AUTH_HDR *auth, char *secret, unsigned char *vector,
		unsigned char seq_nbr) {
    int             secretlen;
    int             totallen;
    unsigned char   calc_digest[AUTH_VECTOR_LEN];
    unsigned char   reply_digest[AUTH_VECTOR_LEN];

    totallen = ntohs(auth->length);

    /* Verify that id (seq. number) matches what we sent */
    if (auth->id != seq_nbr) {
        return EPROTO;
    }

    /* Verify the reply digest */
    memcpy((char*)reply_digest,(char*)auth->vector,AUTH_VECTOR_LEN);
    memcpy((char*)auth->vector,(char*)vector,AUTH_VECTOR_LEN);
    secretlen = strlen (secret);
    memcpy((char*)auth + totallen, secret, secretlen);
    md5_calc (calc_digest, (char*) auth, totallen + secretlen);

    if (memcmp ((char*)reply_digest, (char*)calc_digest,
            AUTH_VECTOR_LEN) != 0) {
        return EPROTO;
    }

    switch (auth->code) {
 	case PW_AUTHENTICATION_ACK: 
	  return -1;		// User Authenticated
	  break;
	case PW_AUTHENTICATION_REJECT:
	  return -2;		// Access Denied
	  break;
    	default:
          return EPROTO;
    }
}

void random_vector(u_char *vector) {
	int	randno;
	int	i;

	srand(time(0));
	for(i = 0;i < AUTH_VECTOR_LEN;) {
		randno = rand();
		memcpy(vector, &randno, sizeof(int));
		vector += sizeof(int);
		i += sizeof(int);
	}
}
